<?php

namespace MeuMouse\Flexify_Checkout\Admin;

use MeuMouse\Flexify_Checkout\Core\Helpers;
use MeuMouse\Flexify_Checkout\API\License;

// Exit if accessed directly.
defined('ABSPATH') || exit;

/**
 * Class to handle plugin admin panel objects and functions
 * 
 * @since 1.0.0
 * @version 5.0.0
 * @package MeuMouse.com
 */
class Admin_Options {

    /**
     * Construct function
     *
     * @since 1.0.0
     * @version 5.0.0
     * @return void
     */
    public function __construct() {
        // set default options
        add_action( 'admin_init', array( $this, 'set_default_options' ) );

        // set default checkout fields options
        add_action( 'admin_init', array( $this, 'set_checkout_step_fields' ) );

        // handle for billing country admin notice
        add_action( 'woocommerce_checkout_init', array( __CLASS__, 'check_billing_country_field' ) );
        add_action( 'admin_notices', array( __CLASS__, 'show_billing_country_warning' ) );
        add_action( 'admin_footer', array( __CLASS__, 'dismiss_billing_country_warning_script' ) );

        // display notice when not has [woocommerce_checkout] shortcode
        add_action( 'admin_notices', array( __CLASS__, 'check_for_checkout_shortcode' ) );

        // display notice when not has PHP gd extension
        add_action( 'admin_notices', array( __CLASS__, 'missing_gd_extension_notice' ) );
    }


    /**
     * Gets the items from the array and inserts them into the option if it is empty,
     * or adds new items with default value to the option
     * 
     * @since 2.3.0
     * @version 5.0.0
     * @return void
     */
    public function set_default_options() {
        $default_options = ( new Default_Options() )->set_default_data_options();
        $get_options = get_option('flexify_checkout_settings', array());

        // if empty settings
        if ( empty( $get_options ) ) {
            update_option( 'flexify_checkout_settings', $default_options );
        } else {
            // iterate for each plugin settings
            foreach ( $get_options as $option => $value ) {
                // iterate for each default settings
                foreach ( $default_options as $index => $option_value ) {
                    if ( ! isset( $get_options[$index] ) ) {
                        $get_options[$index] = $option_value;
                    }
                }
            }

            update_option( 'flexify_checkout_settings', $get_options );
        }
    }


    /**
     * Set default options checkout fields
     * 
     * @since 3.0.0
     * @version 5.0.0
     * @return void
     */
    public function set_checkout_step_fields() {
        $default_options = new Default_Options();
        $get_fields = $default_options->get_native_checkout_fields();
        $get_field_options = maybe_unserialize( get_option('flexify_checkout_step_fields', array()) );

        // create options if array is empty
        if ( empty( $get_field_options ) ) {
            $fields = array();

            foreach ( $get_fields as $key => $value ) {
                $fields[$key] = $value;
            }

            update_option('flexify_checkout_step_fields', maybe_serialize( $fields ) );
        } else {
            foreach ( $get_fields as $key => $value ) {
                if ( ! isset( $get_field_options[$key] ) ) {
                    $get_field_options[$key] = $value;
                }
            }

            update_option( 'flexify_checkout_step_fields', maybe_serialize( $get_field_options ) );
        }

        /**
         * Add integration with Brazilian Market on WooCommerce plugin
         * 
         * @since 1.0.0
         */
        if ( class_exists('Extra_Checkout_Fields_For_Brazil') && ! isset( $get_field_options['billing_cpf'] ) ) {
            $get_field_options = maybe_unserialize( $get_field_options );

            // Add Brazilian Market on WooCommerce fields to existing options
            $wcbcf_fields = $default_options->get_brazilian_checkout_fields();
            $get_field_options = array_merge( $get_field_options, $wcbcf_fields );

            update_option( 'flexify_checkout_step_fields', maybe_serialize( $get_field_options ) );
        }
    }


    /**
     * Checks if the option exists and returns the indicated array item
     * 
     * @since 1.0.0
     * @version 5.0.0
     * @param $key | Array key
     * @return mixed | string or false
     */
    public static function get_setting( $key ) {
        $default_options = get_option( 'flexify_checkout_settings', array() );

        // check if array key exists and return key
        if ( isset( $default_options[$key] ) ) {
            return $default_options[$key];
        }

        return false;
    }


    /**
     * Check if billing country is disabled on checkout
     * 
     * @since 3.7.3
     * @return void
     */
    public static function check_billing_country_field() {
        $checkout_fields = WC()->checkout()->get_checkout_fields();
        $is_disabled = empty( $checkout_fields['billing']['billing_country'] ) || $checkout_fields['billing']['billing_country']['required'] === false;

        update_option( 'billing_country_field_disabled', $is_disabled );
    }


    /**
     * Display admin notice when billing country field is disabled
    * 
    * @since 3.7.3
    * @return void
    */
    public static function show_billing_country_warning() {
        $is_disabled = get_option('billing_country_field_disabled');
        $hide_notice = get_user_meta( get_current_user_id(), 'hide_billing_country_notice', true );

        if ( $is_disabled && ! $hide_notice ) {
            $class = 'notice notice-error is-dismissible';
            $message = esc_html__( 'O campo País na finalização de compras está desativado, verifique se seu gateway de pagamentos depende deste campo para não receber o erro "Informe um endereço para continuar com sua compra."', 'flexify-checkout-for-woocommerce' );
            
            printf( '<div id="billing-country-warning" class="%1$s"><p>%2$s</p></div>', esc_attr( $class ), $message );
        }
    }


    /**
     * Send action on dismiss notice for not display
     * 
     * @since 3.7.3
     * @return void
     */
    public static function dismiss_billing_country_warning_script() {
        ?>
        <script type="text/javascript">
            jQuery(document).on('click', '#billing-country-warning .notice-dismiss', function() {
                jQuery.ajax({
                    url: ajaxurl,
                    type: 'POST',
                    data: {
                        action: 'dismiss_billing_country_warning',
                    }
                });
            });
        </script>
        <?php
    }


    /**
	 * Display error message on WooCommerce checkout page if shortcode is missing
	 * 
	 * @since 4.5.0
	 * @return void
	 */
	public static function check_for_checkout_shortcode() {
		if ( ! Helpers::has_shortcode_checkout() ) {
			$class = 'notice notice-error is-dismissible';
			$message = __( 'O Flexify Checkout depende do shortcode [woocommerce_checkout] na página de finalização de compras para funcionar corretamente.', 'flexify-checkout-for-woocommerce' );

			printf( '<div class="%1$s"><p>%2$s</p></div>', esc_attr( $class ), $message );
		}
	}
  

    /**
	 * Display error message when PHP extensionn gd is missing
	 * 
	 * @since 4.5.0
	 * @return void
	 */
	public static function missing_gd_extension_notice() {
		if ( ! extension_loaded('gd') && Admin_Options::get_setting('enable_inter_bank_pix_api') === 'yes' ) {
			$class = 'notice notice-error is-dismissible';
			$message = __( 'A extensão GD está desativada, e é necessária para gerar o QR Code do Pix. Ative-a em sua hospedagem para habilitar esse recurso.', 'flexify-checkout-for-woocommerce' );

			printf( '<div class="%1$s"><p>%2$s</p></div>', esc_attr( $class ), $message );
		}
	}
}