<?php
function atlt_render_settings_page() {
    $text_domain = 'loco-translate-addon';
    
    // Define APIs configuration
    $apis = [
        'gemini' => [
            'name' => 'Gemini',
            'docs_url' => 'https://locoaddon.com/docs/pro-plugin/how-to-use-gemini-ai-to-translate-plugins-or-themes/generate-gemini-api-key/'
        ],
        'openai' => [
            'name' => 'OpenAI',
            'docs_url' => 'https://platform.openai.com/api-keys'
        ]
    ];
    ?>
    <div class="atlt-dashboard-settings">
        <div class="atlt-dashboard-settings-container">
            <div class="header">
                <h1><?php _e('Loco Addon Settings', $text_domain); ?></h1>
            </div>  
            <p class="description">
                <?php _e('Configure your settings for the Loco Addon to optimize your translation experience. Start by entering your <a href="' . admin_url('admin.php?page=loco-atlt-dashboard&tab=license') . '">license key</a>. Once it\'s activated, you\'ll be able to add your Gemini or OpenAI API keys and manage your preferences for seamless integration.', $text_domain); ?>
            </p>
            <div class="atlt-dashboard-api-settings-container">
                <div class="atlt-dashboard-api-settings">
                    <?php foreach ($apis as $key => $api): ?>
                        <label for="<?php echo esc_attr($key); ?>-api"><?php printf(__('Add %s API key', $text_domain), esc_html($api['name'])); ?></label>
                        <input type="text" id="<?php echo esc_attr($key); ?>-api" placeholder="xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx" disabled>
                        <?php
                        printf(
                            __('%s to See How to Generate %s API Key', $text_domain),
                            '<a href="' . esc_url($api['docs_url']) . '" target="_blank">' . esc_html__('Click Here', $text_domain) . '</a>',
                            esc_html($api['name'])
                        );
                    endforeach; ?>

                    <div class="atlt-dashboard-save-btn-container">
                        <button disabled class="button button-primary"><?php _e('Save', $text_domain); ?></button>
                    </div>
                </div>
            </div>
        </div>
        <div class="atlt-dashboard-geminiAPIkey">
            <h3>Rate Limits of Free Gemini API Key</h3>
            <ul>
                <li><strong>15 RPM</strong>: This API Key allows a maximum of 15 requests per minute</li>
                <li><strong>1 million TPM</strong>: With this API Key, you can process up to 1 million tokens per minute</li>
                <li><strong>1,500 RPD</strong>: To ensure smooth performance, it allows up to 1,500 requests per day</li>
            </ul>
        </div> 
    </div>
    <?php
}

function atlt_render_settings_page_pro() {
    $text_domain = 'loco-translate-addon';
    
    // Move API configuration to a separate function for better organization
    $apis = atlt_get_api_configurations();
    
    // Handle form submission early
    if (atlt_check_form_submission()) {
        atlt_handle_api_key_submission();
    }
    
    atlt_render_settings_page_html($apis, $text_domain);
}

function atlt_get_api_configurations() {
    return [
        'gemini' => [
            'name' => 'Gemini',
            'option_key' => 'LocoAutomaticTranslateAddonPro_google_api_key',
            'docs_url' => 'https://locoaddon.com/docs/pro-plugin/how-to-use-gemini-ai-to-translate-plugins-or-themes/generate-gemini-api-key/',
            'value' => get_option('LocoAutomaticTranslateAddonPro_google_api_key', '')
        ],
        'openai' => [
            'name' => 'OpenAI',
            'option_key' => 'LocoAutomaticTranslateAddonPro_openai_api_key',
            'docs_url' => 'https://platform.openai.com/api-keys',
            'value' => get_option('LocoAutomaticTranslateAddonPro_openai_api_key', '')
        ]
    ];
}

function atlt_check_form_submission() {
    return $_SERVER['REQUEST_METHOD'] === 'POST' && 
           isset($_POST['nonce']) && 
           wp_verify_nonce($_POST['nonce'], 'api_keys');
}

function atlt_validate_google_api_key($key) {
    if (empty($key)) return false;
    
    if (!preg_match('/^AIza[0-9A-Za-z-_]{35}$/', $key)) {
        atlt_show_admin_notice('error', 'Invalid Gemini API Key.');
        return false;
    }

    $response = wp_remote_post(
        'https://generativelanguage.googleapis.com/v1beta/models/gemini-1.5-flash-latest:generateContent?key=' . $key,
        [
            'headers' => ['Content-Type' => 'application/json'],
            'body' => json_encode(['contents' => [['parts' => [['text' => 'Test']]]]]),
            'method' => 'POST',
            'timeout' => 60,
        ]
    );

    if (is_wp_error($response) || isset(json_decode(wp_remote_retrieve_body($response), true)['error'])) {
        atlt_show_admin_notice('error', 'Invalid Gemini API Key.');
        return false;
    }

    return true;
}

function atlt_validate_openai_api_key($key) {
    if (empty($key)) return false;

    $response = wp_remote_post('https://api.openai.com/v1/chat/completions', [
        'headers' => [
            'Content-Type' => 'application/json',
            'Authorization' => 'Bearer ' . $key,
        ],
        'body' => wp_json_encode([
            'model' => 'gpt-4o-mini',
            'messages' => [['role' => 'user', 'content' => 'Test']]
        ]),
        'method' => 'POST',
    ]);

    if (is_wp_error($response)) {
        atlt_show_admin_notice('error', 'Invalid OpenAI API Key.');
        return false;
    }

    $response_data = json_decode(wp_remote_retrieve_body($response), true);
    if (!empty($response_data['error'])) {
        $error_message = $response_data['error']['message'] ?? 'Invalid OpenAI API Key.';
        atlt_show_admin_notice('error', str_replace('<a ', '<a target="_blank" ', make_clickable($error_message)));
        return false;
    }

    return true;
}

function atlt_show_admin_notice($type, $message) {
    if (!isset($GLOBALS['atlt_admin_notices'])) {
        $GLOBALS['atlt_admin_notices'] = array();
    }
    $GLOBALS['atlt_admin_notices'][] = '<div class="notice notice-' . esc_attr($type) . ' is-dismissible"><p>' . wp_kses($message, ['a' => ['href' => [], 'target' => []]]) . '</p></div>';
}

function atlt_handle_api_key_submission() {
    // Clear any existing notices at the start
    $GLOBALS['atlt_admin_notices'] = array();
    
    if (isset($_POST['reset_gemini_api_key'])) {
        delete_option('LocoAutomaticTranslateAddonPro_google_api_key');
        atlt_show_admin_notice('success', 'Gemini API Key has been removed.');
        return true;
    } 
    
    if (isset($_POST['reset_openai_api_key'])) {
        delete_option('LocoAutomaticTranslateAddonPro_openai_api_key');
        atlt_show_admin_notice('success', 'OpenAI API Key has been removed.');
        return true;
    }
    
    if (isset($_POST['submit_api_keys'])) {
        return atlt_handle_api_key_save();
    }
    
    return false;
}

function atlt_handle_api_key_save() {
    $success = false;
    $any_validation_attempted = false;
    
    if (isset($_POST['LocoAutomaticTranslateAddonPro_google_api_key'])) {
        $new_google_key = sanitize_text_field($_POST['LocoAutomaticTranslateAddonPro_google_api_key']);
        if (!empty($new_google_key)) {
            $any_validation_attempted = true;
            if (atlt_validate_google_api_key($new_google_key)) {
                update_option('LocoAutomaticTranslateAddonPro_google_api_key', $new_google_key);
                $success = true;
            }
        }
    }
    
    if (isset($_POST['LocoAutomaticTranslateAddonPro_openai_api_key'])) {
        $new_openai_key = sanitize_text_field($_POST['LocoAutomaticTranslateAddonPro_openai_api_key']);
        if (!empty($new_openai_key)) {
            $any_validation_attempted = true;
            if (atlt_validate_openai_api_key($new_openai_key)) {
                update_option('LocoAutomaticTranslateAddonPro_openai_api_key', $new_openai_key);
                $success = true;
            }
        }
    }
    
    if ($success) {
        atlt_show_admin_notice('success', 'API keys saved successfully.');
        return true;
    } elseif ($any_validation_attempted && !isset($GLOBALS['atlt_admin_notices'])) {
        // Only show generic error if we attempted validation and no specific error was set
        atlt_show_admin_notice('error', 'Please enter a valid API key.');
    }
    
    return false;
}

function atlt_render_settings_page_html($apis, $text_domain) {
    // Process form submission before rendering
    $form_processed = false;
    if (atlt_check_form_submission()) {
        $form_processed = atlt_handle_api_key_submission();
    }
    
    // Refresh API values after form processing
    if ($form_processed) {
        $apis = atlt_get_api_configurations();
    }
    
    ?>
    <div class="atlt-dashboard-settings">
        <div class="atlt-dashboard-settings-container">
            <?php
            // Show notices at the top of the container
            if (isset($GLOBALS['atlt_admin_notices'])) {
                foreach ($GLOBALS['atlt_admin_notices'] as $notice) {
                    echo $notice;
                }
            }
            ?>
            <div class="header">
                <h1><?php _e('Loco Addon Settings', $text_domain); ?></h1>
            </div>
            
            <p class="description">
                <?php _e('Configure your settings for the Loco Addon to optimize your translation experience. Enter your API keys and manage your preferences for seamless integration.', $text_domain); ?>
            </p>

            <div class="atlt-dashboard-api-settings-container">
                <div class="atlt-dashboard-api-settings">
                    <form method="post">
                        <?php wp_nonce_field('api_keys', 'nonce'); ?>
                        
                        <?php foreach ($apis as $key => $api): 
                            $has_key = !empty($api['value']);
                            $masked_value = $has_key ? 
                                esc_attr(substr($api['value'], 0, 8) . str_repeat('*', 24) . substr($api['value'], -8) . ' ✅') : 
                                ''; 
                        ?>
                            <label for="<?php echo esc_attr($key); ?>-api" class="api-settings-label">
                                <?php printf(__('Add %s API key', $text_domain), esc_html($api['name'])); ?>
                            </label>
                            <input type="text" 
                                id="<?php echo esc_attr($key); ?>-api" 
                                name="<?php echo esc_attr($api['option_key']); ?>" 
                                value="<?php echo $masked_value; ?>" 
                                placeholder="xxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxxx"
                                <?php echo $has_key ? 'disabled' : ''; ?>>
                            
                            <?php if ($has_key): ?>
                                <button type="submit" name="reset_<?php echo esc_attr($key); ?>_api_key" class="button button-primary">
                                    <?php _e('Reset', $text_domain); ?>
                                </button>
                            <?php else: ?>
                                <?php
                                printf(
                                    __('%s to See How to Generate %s API Key', $text_domain),
                                    '<a href="' . esc_url($api['docs_url']) . '" target="_blank">' . esc_html__('Click Here', $text_domain) . '</a>',
                                    esc_html($api['name'])
                                );
                                ?>
                            <?php endif; ?>
                        <?php endforeach; ?>

                        <?php if (!atlt_all_apis_configured($apis)): ?>
                            <div class="atlt-dashboard-save-btn-container">
                                <button type="submit" name="submit_api_keys" class="button button-primary">
                                    <?php _e('Save', $text_domain); ?>
                                </button>
                            </div>
                        <?php endif; ?>
                    </form>
                </div>
            </div>
        </div>
        <div class="atlt-dashboard-geminiAPIkey">
            <h3>Rate Limits of Free Gemini API Key</h3>
            <ul>
                <li><strong>15 RPM</strong>: This API Key allows a maximum of 15 requests per minute</li>
                <li><strong>1 million TPM</strong>: With this API Key, you can process up to 1 million tokens per minute</li>
                <li><strong>1,500 RPD</strong>: To ensure smooth performance, it allows up to 1,500 requests per day</li>
            </ul>
        </div> 
    </div>
    <?php
}

function atlt_all_apis_configured($apis) {
    foreach ($apis as $api) {
        if (empty($api['value'])) return false;
    }
    return true;
}


