<?php
/**
 * Class Loco_Automatic_Translate_Addon_Pro\AI_Translate\Services\Authentication\API_Key_Authentication
 */

namespace Loco_Automatic_Translate_Addon_Pro\AI_Translate\Services\Authentication;

use Loco_Automatic_Translate_Addon_Pro\AI_Translate\Services\Contracts\Authentication;
use Felix_Arntz\WP_OOP_Plugin_Lib\HTTP\Contracts\Request;

/**
 * Class that represents an API key.
 */
final class API_Key_Authentication implements Authentication {

	/**
	 * The API key.
	 *
	 * @var string
	 */
	private $api_key;

	/**
	 * The HTTP header to use for the API key.
	 *
	 * @var string
	 */
	private $header_name = 'Authorization';

	/**
	 * Constructor.
	 *
	 * @param string $api_key The API key.
	 */
	public function __construct( string $api_key ) {
		$this->api_key = $api_key;
	}

	/**
	 * Authenticates the given request with the credentials.
	 *
	 * @param Request $request The request instance. Updated in place.
	 */
	public function authenticate( Request $request ): void {
		if ( 'authorization' === strtolower( $this->header_name ) ) {
			$request->add_header( $this->header_name, 'Bearer ' . $this->api_key );
		} else {
			$request->add_header( $this->header_name, $this->api_key );
		}
	}

	/**
	 * Sets the header name to use to add the credentials to a request.
	 *
	 * @param string $header_name The header name.
	 */
	public function set_header_name( string $header_name ): void {
		$this->header_name = $header_name;
	}

	/**
	 * Returns the option definitions needed to store the credentials.
	 *
	 * @param string $service_slug The service slug.
	 * @return array<string, array<string, mixed>> The option definitions.
	 */
	public static function get_option_definitions( string $service_slug ): array {
		$option_slug = sprintf( 'LocoAutomaticTranslateAddonPro_%s_api_key', $service_slug );

		return array(
			$option_slug => array(
				'type'         => 'string',
				'default'      => '',
				'show_in_rest' => true,
				'autoload'     => true,
			),
		);
	}
}
